package cz.drg.clasificator.util;

import cz.drg.clasificator.exception.ShutdownException;
import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.util.ArrayList;
import java.util.List;
import java.util.Scanner;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipInputStream;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import org.dmg.pmml.DefineFunction;
import org.dmg.pmml.InlineTable;
import org.dmg.pmml.MapValues;
import org.dmg.pmml.PMML;
import org.dmg.pmml.Row;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

/**
 *
 * @author jirasek
 */
public class UnmarshallOptimized {

    private ExecutorService pool;
    private static final String SIMPLIFIED_PMML_FOLDERPATH = "./log/";
    private static final String SIMPLIFIED_PMML_FILEPATH = SIMPLIFIED_PMML_FOLDERPATH + "editedModelCopy" + System.currentTimeMillis() + ".xml";

    public UnmarshallOptimized() {
        int availableProcessors = Runtime.getRuntime().availableProcessors();
        pool = new ThreadPoolExecutor(availableProcessors, availableProcessors, 0, TimeUnit.MILLISECONDS, new LinkedBlockingQueue<>());
    }

    public PMML unmarshal(ZipFile pmmlFile) {

        PMML pmml = startOptimizedUnmarshaling(pmmlFile);

        pool.shutdown();

        try {
            Files.deleteIfExists(Paths.get(SIMPLIFIED_PMML_FILEPATH));
            FilesCache.freeRemovedFile(SIMPLIFIED_PMML_FILEPATH);
        } catch (IOException ex) {
            Logger.getLogger(UnmarshallOptimized.class.getName()).log(Level.SEVERE, null, ex);
        }

        return pmml;

    }

    private PMML startOptimizedUnmarshaling(ZipFile pmmlFile) {

        //create simplified pmml file where inline tables in DefineFunctions are replaced
        //with empty tables and get their list
        List<String> defineFunctionsWithInlineTables = createSimplifiedPmmlFile(pmmlFile);
        //submit all DefineFunctions with inline tables for paralel processing
        List<Future> paralelUnmarshalTasks = startParalelProcessing(defineFunctionsWithInlineTables);
        //unmarshal simplified PMML
        PMML pmml = new UnmarshallNormaly().unmarshal(new File(SIMPLIFIED_PMML_FILEPATH));
        //wait until all DefineFunctions are parsed
        waitTillAllTasksFinished(paralelUnmarshalTasks);
        //update the unmarshaled simplified PMML
        replaceDefineFunctions(pmml, paralelUnmarshalTasks);

        return pmml;
    }

    /**
     * Creates empty pmml file, that will be used as a target for copying
     * simplified version of original pmml file.
     */
    private void ensureSimplifiedPmmlFile() {

        File editedPmmlCopy = new File(SIMPLIFIED_PMML_FILEPATH);
        File logFolder = new File(SIMPLIFIED_PMML_FOLDERPATH);
        editedPmmlCopy.delete();
        FilesCache.freeRemovedFile(SIMPLIFIED_PMML_FILEPATH);

        try {

            logFolder.mkdirs();
            editedPmmlCopy.createNewFile();
            FilesCache.addCreatedFile(SIMPLIFIED_PMML_FILEPATH);

        } catch (IOException ex) {
            Logger.getLogger(UnmarshallOptimized.class.getName()).log(Level.SEVERE, null, ex);
            throw new ShutdownException(String.format(Constants.ERR_CANNOT_CREATE_FILE, SIMPLIFIED_PMML_FILEPATH));
        }

    }

    /**
     * This method create edited copy of the original pmml file, where all
     * DefineFunctions inline tables are replaced with empty tables. This
     * prevents the normal PMML unmarshaling to be slowed down, because of these
     * long tables. Instead these table are processed later in paralel to ensure
     * the maximum possible speed.
     *
     * @param pmmlFile
     * @return list of xml representations of DefineFunctions from original file
     * with inline tables in them
     */
    private List<String> createSimplifiedPmmlFile(ZipFile pmmlFile) {

        List<String> defineFunctionsWithInlineTables = new ArrayList<>();

        ensureSimplifiedPmmlFile();

        ZipEntry pmmlModel = pmmlFile.entries().nextElement();
        //first there should be a copy created so that the original file is untouched
        try(BufferedReader br = new BufferedReader(new InputStreamReader(pmmlFile.getInputStream(pmmlModel)))) {
            
            List<String> readLines = br.lines().collect(Collectors.toList());
            
            boolean defineFunction = false;
            boolean hasInlineTable = false;
            StringBuilder sb = null;

            List<String> batchWrite = new ArrayList<>();

            for (String row : readLines) {

                if (!defineFunction) {
                    batchWrite.add(row);
                }
                //parse all rows related to DefineFunction
                //if it has InlineTable store all the rows related to this DefineFunction for later processing

                //IMPROVEMENT check index of the text and add just text after not before it to avoid including some
                //rogue xml breaking validity of xml
                if (row.contains("<DefineFunction")) {
                    //start of new DefineFunction
                    defineFunction = true;
                    //xmlns tag needs to be added so marshaler knows the structure
                    sb = new StringBuilder(row.replaceFirst(">", " xmlns=\"http://www.dmg.org/PMML-4_3\">"));
                    sb.append("\n");

                    //write batch of all previous rows and clear
                    //remove last added row "<DefineFunction"
                    batchWrite.remove(batchWrite.size() - 1);
                    Files.write(Paths.get(SIMPLIFIED_PMML_FILEPATH), batchWrite, StandardOpenOption.APPEND);
                    batchWrite.clear();
                } //IMPROVEMENT check index of the text and add just text before not after it to avoid including some
                //rogue xml breaking validity of xml
                else if (row.contains("</DefineFunction>")) {
                    //end of the DefineFunction
                    defineFunction = false;
                    sb.append(row);

                    if (hasInlineTable) {

                        //this DefineFunction needs to be replaced by DefineFunction with empty InlineTable
                        //for full PMML unmarshaling and the contents needs to be processed separately and added later
                        //this will copy whole content of DefineFunction
                        //Files.write(sb.toString().getBytes(StandardCharsets.UTF_8), editedPmmlCopy);
                        String defineFunctionXml = sb.toString();
                        String replacement = emptyInlineTable(defineFunctionXml);

                        //lets just keep first line of inlineTable row there for now and replace later
                        //with full proccessed InlineTable object
                        Files.write(Paths.get(SIMPLIFIED_PMML_FILEPATH), replacement.getBytes(), StandardOpenOption.APPEND);

                        defineFunctionsWithInlineTables.add(sb.toString());
                    } else {
                        Files.write(Paths.get(SIMPLIFIED_PMML_FILEPATH), sb.toString().getBytes(), StandardOpenOption.APPEND);
                    }
                    hasInlineTable = false;
                    sb = null;
                } else if (defineFunction) {
                    //some content of DefineFunction
                    sb.append(row);
                }

                if (row.contains("<InlineTable>")) {
                    hasInlineTable = true;
                }

            }

            //write batch of all previous rows and clear
            Files.write(Paths.get(SIMPLIFIED_PMML_FILEPATH), batchWrite, StandardOpenOption.APPEND);
            batchWrite.clear();

        } catch (IOException ex) {
            Logger.getLogger(UnmarshallOptimized.class.getName()).log(Level.SEVERE, null, ex);
        }

        return defineFunctionsWithInlineTables;
    }

    private String emptyInlineTable(String defineFunctionXml) {

        int endOfFirstRow = defineFunctionXml.indexOf("</row>");
        int endOfInlineTable = defineFunctionXml.indexOf("</InlineTable>");

        String firstLineIncluded = defineFunctionXml.substring(0, endOfFirstRow + 6);
        String closures = defineFunctionXml.substring(endOfInlineTable);
        String replacement = firstLineIncluded.concat(closures);

        return replacement;
    }

    private void waitTillAllTasksFinished(List<Future> paralelUnmarshalTasks) {
        //wait for all DefineFunction tasks to finish
        boolean finished = false;
        while (!finished) {

            try {
                //no need to check more often
                Thread.sleep(50);

                finished = true;
                for (Future paralelUnmarshalTask : paralelUnmarshalTasks) {
                    if (!paralelUnmarshalTask.isDone()) {
                        finished = false;
                        break;
                    }
                }
            } catch (InterruptedException ex) {
                Logger.getLogger(UnmarshallOptimized.class.getName()).log(Level.SEVERE, null, ex);
            }

        }
    }

    private void replaceDefineFunctions(PMML pmml, List<Future> paralelUnmarshalTasks) {

        List<DefineFunction> defineFunctions = pmml.getTransformationDictionary().getDefineFunctions();

        for (Future paralelUnmarshalTask : paralelUnmarshalTasks) {

            try {
                DefineFunction function = (DefineFunction) paralelUnmarshalTask.get();

                //find simlified version of the function
                for (int i = 0; i < defineFunctions.size(); i++) {
                    DefineFunction simpleFunction = defineFunctions.get(i);
                    if (simpleFunction.getName().equals(function.getName())) {
                        //replace the simple version
                        defineFunctions.set(i, function);
                    }
                }
            } catch (InterruptedException | ExecutionException ex) {
                Logger.getLogger(UnmarshallOptimized.class.getName()).log(Level.SEVERE, null, ex);
            }

        }
    }

    private List<Future> startParalelProcessing(List<String> defineFunctionsWithInlineTables) {
        List<Future> paralelUnmarshalTasks = new ArrayList<>();

        for (String defineFunctionsWithInlineTable : defineFunctionsWithInlineTables) {
            paralelUnmarshalTasks.add(pool.submit(new UnmarshallDefineFunction(defineFunctionsWithInlineTable)));
        }

        return paralelUnmarshalTasks;
    }

    private class UnmarshallDefineFunction implements Callable<DefineFunction> {

        private final String xmlData;
        private ExecutorService defineFunctionPool;

        private final String inlineTableTag = "<InlineTable>";
        private final int inlineTableTagLength = inlineTableTag.length();

        private final String inlineTableEndTag = "</InlineTable>";
        private final int inlineTableEndTagLength = inlineTableEndTag.length();

        private final String rowEndTag = "</row>";
        private final int rowEndTagLength = rowEndTag.length();

        public UnmarshallDefineFunction(String xmlData) {
            this.xmlData = xmlData;
        }

        @Override
        public DefineFunction call() throws Exception {

            //long now = System.currentTimeMillis();
            DefineFunction function = null;
            boolean tooLargeInlineTable = false;

            try {

                //split the xmlData to more Threads if necessary
                int inlineTableTagIndex = xmlData.indexOf(inlineTableTag);
                int inlineTableEndTagIndex = xmlData.indexOf(inlineTableEndTag);
                int firstRowEndTagIndex = xmlData.indexOf("</row>");

                int endFirstRowIndexTag = firstRowEndTagIndex + rowEndTagLength;
                int endInlineTableTagIndex = inlineTableTagIndex + inlineTableTagLength;

                int rowLength = endFirstRowIndexTag - endInlineTableTagIndex;

                int totalRowLength = ((inlineTableEndTagIndex + inlineTableEndTagLength) - inlineTableTagIndex) - inlineTableTagLength - inlineTableEndTagLength;
                int estimatedRows = totalRowLength / rowLength;

                if (estimatedRows > 10000) {
                    tooLargeInlineTable = true;

//                    int availableProcessors = Runtime.getRuntime().availableProcessors();
                    int availableProcessors = 1;
                    defineFunctionPool = new ThreadPoolExecutor(availableProcessors, availableProcessors, 0, TimeUnit.MILLISECONDS, new LinkedBlockingQueue<>());

                    List<Future> inlineTableTasks = new ArrayList<>();

                    String withoutInlineTableTag = xmlData.substring(inlineTableTagIndex + inlineTableTagLength, inlineTableEndTagIndex);

                    int previousOffset = 0;
                    int numberOfRowsPerTask = 10000;

                    int lastRowGroup = totalRowLength / (rowLength * numberOfRowsPerTask);

                    for (int i = 0; i <= lastRowGroup; i++) {

                        int beginAt = i * rowLength * numberOfRowsPerTask - previousOffset;
                        int substringTo = (i + 1) * rowLength * numberOfRowsPerTask > withoutInlineTableTag.length() ? withoutInlineTableTag.length() : (i + 1) * rowLength * numberOfRowsPerTask;

                        String rows = withoutInlineTableTag.substring(beginAt, substringTo);
                        //most likely splitted somewhere in the middle of row
                        int lastValidEndRow = rows.lastIndexOf(rowEndTag) + rowEndTagLength;
                        previousOffset = rows.length() - lastValidEndRow;
                        rows = rows.substring(0, lastValidEndRow);

                        String splittedInlineTable = this.inlineTableTag.concat(rows).concat(this.inlineTableEndTag);
                        //add xmlns tag to the root inline table tag for it to be unmarshalable
                        splittedInlineTable = splittedInlineTable.replaceFirst(">", " xmlns=\"http://www.dmg.org/PMML-4_3\">");

                        inlineTableTasks.add(defineFunctionPool.submit(new UnmarshallInlineTable(splittedInlineTable)));

                    }

                    String functionWithoutRows = xmlData.substring(0, endInlineTableTagIndex);
                    functionWithoutRows = functionWithoutRows.concat(xmlData.substring(xmlData.indexOf(inlineTableEndTag)));

                    function = unmarshalData(functionWithoutRows);

                    boolean allFinished = false;
                    while (!allFinished) {

                        Thread.sleep(400);
                        allFinished = true;

                        for (Future inlineTableTask : inlineTableTasks) {
                            if (!inlineTableTask.isDone()) {
                                allFinished = false;
                                break;
                            }
                        }

                    }

                    InlineTable completeTable = new InlineTable();

                    for (Future inlineTableTask : inlineTableTasks) {
                        InlineTable table = (InlineTable) inlineTableTask.get();
                        completeTable.getRows().addAll(table.getRows());
                    }

                    MapValues expression = (MapValues) function.getExpression();
                    expression.setInlineTable(completeTable);

                }
                //getVdgSeverityAppC should have 554341 rows in total

                if (!tooLargeInlineTable) {
                    function = unmarshalData(xmlData);
                }

            } finally {
                if (defineFunctionPool != null) {
                    defineFunctionPool.shutdown();
                }
            }

            return function;
        }

        private DefineFunction unmarshalData(String functionData) {

//            JAXBContext jc;
            DefineFunction function = null;
            try {

//                function = JacksonUtil.read(DefineFunction.class, new ByteArrayInputStream(functionData.getBytes()));
//                PolymorphicTypeValidator ptv = BasicPolymorphicTypeValidator.builder()
//                        .allowIfSubType("org.dmg.pmml.Expression")
//                        //                      .allowIfSubType("org.dmg.pmml.ParameterField")
////                                              .allowIfSubType("java.util.ArrayList")
////                                              .allowIfSubType("java.util.List")
//                        .build();
//                XmlMapper mapper = new XmlMapper();
//                mapper.activateDefaultTyping(ptv);
//mapper.configure(DeserializationFeature.ACCEPT_SINGLE_VALUE_AS_ARRAY, true);
//mapper.configure(DeserializationFeature.UNWRAP_ROOT_VALUE, true);
//                CollectionType listType = mapper.getTypeFactory().constructCollectionType(List.class, ParameterField.class);
//                CollectionType arrayListType = mapper.getTypeFactory().constructCollectionType(ArrayList.class, ParameterField.class);
//                JavaType paramFieldType = mapper.getTypeFactory().constructType(new TypeReference<ParameterField>() {});
//                
//                
//                SimpleModule module = new SimpleModule("configModule", com.fasterxml.jackson.core.Version.unknownVersion());
//                module.addDeserializer(DefineFunction.class, new DeSerializer());
//                mapper.registerModule(module);
                
//                mapper.registerModule(new SimpleModule() {
//                    @Override
//                    public void setupModule(Module.SetupContext context) {
//                        super.setupModule(context); 
//                        context.addBeanDeserializerModifier(new BeanDeserializerModifier() {
//                            @Override
//                            public JsonDeserializer<?> modifyCollectionDeserializer(DeserializationConfig config, CollectionType type, BeanDescription beanDesc, JsonDeserializer<?> deserializer) {
//                                
//                                
//                                    System.out.println("Modifying deserializer for "+type+ " "+ type.equals(arrayListType) + " "+ type.equals(listType));
//                                
//                                    if(type.equals(arrayListType)){
//                                        System.out.println("New deserializer");
//                                        return new ListDeSerializer();
//                                    }
//                                    
//                                return deserializer;
//                                
//                            }
//                            
//                        });
//                        
//                        context.addBeanDeserializerModifier(new BeanDeserializerModifier(){
//                            @Override
//                            public JsonDeserializer<?> modifyDeserializer(DeserializationConfig config, BeanDescription beanDesc, JsonDeserializer<?> deserializer) {
//                                
//                                System.out.println("AAAModifying deserializer for "+beanDesc.getType()+" "+beanDesc.getType().equals(paramFieldType) );
//                                
//                                return deserializer;
//                            }
//                            
//                        });
//                    }
//                    
//                });
//                            XmlMapper mapper = XmlMapper.builder()
//                .defaultUseWrapper(false)
//                .configure(SerializationFeature.INDENT_OUTPUT, true)
//                .build();
//                            mapper.setAnnotationIntrospector(AnnotationIntrospector.pair(new JacksonAnnotationIntrospector(), new JaxbAnnotationIntrospector(mapper.getTypeFactory())));
////                            mapper.setAnnotationIntrospector(AnnotationIntrospector.pair(new JaxbAnnotationIntrospector(mapper.getTypeFactory()), new JacksonAnnotationIntrospector()));
//                    mapper.configure(MapperFeature.ACCEPT_CASE_INSENSITIVE_PROPERTIES, true);
                function = JacksonUtil.mapper().readValue(functionData, DefineFunction.class);

//                jc = JAXBContext.newInstance(DefineFunction.class);
//                Unmarshaller unmarshaller = jc.createUnmarshaller();
//                
//                StringReader reader = new StringReader(functionData);
//                function = (DefineFunction)unmarshaller.unmarshal(reader);
            } //            catch (JAXBException ex) {
            //                Logger.getLogger(UnmarshallNormaly.class.getName()).log(Level.SEVERE, null, ex);
            //            } 
            catch (IOException ex) {
                Logger.getLogger(UnmarshallOptimized.class.getName()).log(Level.SEVERE, null, ex);
            }
            return function;
        }

    }
    
//     class WrapperDeserializer extends JsonDeserializer<List<?>> implements ContextualDeserializer {
//    private JavaType valueType;
//
//    @Override
//    public JsonDeserializer<?> createContextual(DeserializationContext ctxt, BeanProperty property) throws JsonMappingException {
//        JavaType wrapperType = property.getType();
//        JavaType valueType = wrapperType.containedType(0);
//        WrapperDeserializer deserializer = new WrapperDeserializer();
//        deserializer.valueType = valueType;
//        return deserializer;
//    }
//
//    @Override
//    public Wrapper<?> deserialize(JsonParser parser, DeserializationContext ctxt) throws IOException {
//        Wrapper<?> wrapper = new Wrapper<>();
//        wrapper.value = ctxt.readValue(parser, valueType);
//        return wrapper;
//    }
//}

//    class ListDeSerializer extends JsonDeserializer<List<?>> implements ContextualDeserializer {
//boolean processNextToken = true;
//
//private JavaType valueType;
//
//        @Override
//        public List<?> deserialize(JsonParser p, DeserializationContext dc) throws IOException, JsonProcessingException {
//            JsonToken nextToken = p.nextToken();
//            while (processNextToken) {
//
//                System.out.println("token:" + nextToken + "|" + p.getText() + " id:" + nextToken.id());
//            }
//            return new ArrayList();
//        }
//
//        @Override
//        public JsonDeserializer<?> createContextual(DeserializationContext dc, BeanProperty bp) throws JsonMappingException {
//            JavaType wrapperType = bp.getType();
//            JavaType valueType = wrapperType.containedType(0);
//            ListDeSerializer deserializer = new ListDeSerializer();
//            deserializer.valueType = valueType;
//            return deserializer;
//        }
//        
//    }
    
    
//    class DeSerializer extends StdDeserializer<DefineFunction> {
//
//        private static final String FUNCTION_TYPE = "DefineFunction";
//        private static final String PARAM_FIELD_TYPE = "ParameterField";
//
//        String currentDataObjectType = FUNCTION_TYPE;
//        ParameterField currentParamField = null;
//        DefineFunction defineFunction = new DefineFunction();
//        boolean processNextToken = true;
//
//        protected DeSerializer() {
//            super(DefineFunction.class);
//        }
//
//        @Override
//        public DefineFunction deserialize(JsonParser p, DeserializationContext ctxt) throws IOException, JsonProcessingException {
//            // use p.getText() and p.nextToken to navigate through the xml and construct Person object
//            reset();
//
//            TreeNode readValueAsTree = p.readValueAsTree();
//            ObjectMapper mapper = new ObjectMapper();
////                    .enable(DeserializationFeature.ACCEPT_SINGLE_VALUE_AS_ARRAY);
//                            PolymorphicTypeValidator ptv = BasicPolymorphicTypeValidator.builder()
////                        .allowIfSubType("org.dmg.pmml.Expression")
//                        //                      .allowIfSubType("org.dmg.pmml.ParameterField")
////                                              .allowIfSubType("java.util.ArrayList")
////                                              .allowIfSubType("java.util.List")
//                        .build();
////                            mapper.activateDefaultTyping(ptv);
//            mapper = new ObjectMapper()
//                    .enable(DeserializationFeature.ACCEPT_SINGLE_VALUE_AS_ARRAY)
//                    .activateDefaultTyping(BasicPolymorphicTypeValidator.builder().build());
//            DefineFunction readValue = mapper.readValue(readValueAsTree.toString(), DefineFunction.class);
//            
//            JsonToken nextToken = p.nextToken();
//            
//            while (processNextToken) {
//
//                System.out.println("token:" + nextToken + "|" + p.getText() + " id:" + nextToken.id());
//
//                switch (currentDataObjectType) {
//
//                    case FUNCTION_TYPE: {
//                        handleDefineFunction(p, nextToken);
//                        break;
//                    }
//
//                    case PARAM_FIELD_TYPE: {
//                        handleParameterField(p, nextToken);
//                        break;
//                    }
//
//                    default: {
//                        System.out.println("Unknown data type: " + currentDataObjectType + " using Jackson default.");
//                        break;
//                    }
//
//                }
//
//                try {
//                    CompletableFuture<JsonToken> future = new CompletableFuture<>();
//                    Executors.newCachedThreadPool().submit(() -> future.complete(p.nextToken()));
//                    nextToken = future.get(3, TimeUnit.SECONDS);
//                } catch (InterruptedException ex) {
//                    Logger.getLogger(UnmarshallOptimized.class.getName()).log(Level.SEVERE, null, ex);
//                } catch (ExecutionException ex) {
//                    Logger.getLogger(UnmarshallOptimized.class.getName()).log(Level.SEVERE, null, ex);
//                } catch (TimeoutException ex) {
//                    System.out.println("no result this time end of stream");
//                }
//
//            }
//
//            return defineFunction;
//
//        }
//
//        private void reset() {
//            currentDataObjectType = FUNCTION_TYPE;
//            currentParamField = null;
//            defineFunction = new DefineFunction();
//            processNextToken = true;
//        }

//        private void handleDefineFunction(JsonParser p, JsonToken nextToken) throws IOException {
//            switch (nextToken.toString()) {
//                case "FIELD_NAME": {
//
//                    switch (p.getText()) {
//                        case "name": {
//                            JsonToken value = p.nextToken();
//                            if (value != JsonToken.VALUE_STRING) {
//                                System.out.println("Unexpected json token type after 'name': " + value + " text:" + p.getText());
//                                break;
//                            }
//                            defineFunction.setName(p.getText());
//                            break;
//                        }
//                        case "optype": {
//                            JsonToken value = p.nextToken();
//                            if (value != JsonToken.VALUE_STRING) {
//                                System.out.println("Unexpected json token type after 'optype': " + value + " text:" + p.getText());
//                                break;
//                            }
//                            defineFunction.setOpType(OpType.fromValue(p.getText()));
//                            break;
//                        }
//                        case "dataType": {
//                            JsonToken value = p.nextToken();
//                            if (value != JsonToken.VALUE_STRING) {
//                                System.out.println("Unexpected json token type after 'dataType': " + value + " text:" + p.getText());
//                                break;
//                            }
//                            defineFunction.setDataType(DataType.fromValue(p.getText()));
//                            break;
//                        }
//
//                        case PARAM_FIELD_TYPE: {
//                            System.out.println("New param field.");
//                            currentDataObjectType = PARAM_FIELD_TYPE;
//                            currentParamField = new ParameterField();
//                            break;
//                        }
//                        default: {
//                            System.out.println("Unknown parameter: " + p.getText() + " text:" + p.getText());
//                            break;
//                        }
//
//                    }
//                    break;
//                }
//
//                case "END_OBJECT": {
//                    System.out.println("End of DefineFunction deserializer.");
//                    processNextToken = false;
//                    break;
//
//                }
//                default: {
//                    System.out.println("Unknown token: " + nextToken + " text:" + p.getText());
//                    break;
//                }
//            }
//
//        }
//
//        private void handleParameterField(JsonParser p, JsonToken nextToken) throws IOException {
//            switch (nextToken.toString()) {
//                case "FIELD_NAME": {
//
//                    switch (p.getText()) {
//                        case "name": {
//                            JsonToken value = p.nextToken();
//                            if (value != JsonToken.VALUE_STRING) {
//                                System.out.println("Unexpected json token type: " + value + " text:" + p.getText() + " for token: " + nextToken);
//                                break;
//                            }
//                            currentParamField.setName(FieldName.create(p.getText()));
//                            break;
//                        }
//                        case "optype": {
//                            JsonToken value = p.nextToken();
//                            if (value != JsonToken.VALUE_STRING) {
//                                System.out.println("Unexpected json token type: " + value + " text:" + p.getText() + " for token: " + nextToken);
//                                break;
//                            }
//                            currentParamField.setOpType(OpType.fromValue(p.getText()));
//                        }
//                        case "dataType": {
//                            JsonToken value = p.nextToken();
//                            if (value != JsonToken.VALUE_STRING) {
//                                System.out.println("Unexpected json token type: " + value + " text:" + p.getText() + " for token: " + nextToken);
//                                break;
//                            }
//                            currentParamField.setDataType(DataType.fromValue(p.getText()));
//                            break;
//                        }
//
//                        default: {
//                            System.out.println("Unknown parameter: " + nextToken + " text:" + p.getText());
//                            break;
//                        }
//
//                    }
//                    break;
//                }
//
//                case PARAM_FIELD_TYPE: {
//                    currentDataObjectType = PARAM_FIELD_TYPE;
//                    break;
//                }
//
//                case "END_OBJECT": {
//                    System.out.println("Close param field");
//                    currentDataObjectType = FUNCTION_TYPE;
//                    defineFunction.addParameterFields(currentParamField);
//                    currentParamField = null;
//                    break;
//                }
//
//                default: {
//                    System.out.println("Unknown token: " + nextToken + " text:" + p.getText());
//                    break;
//                }
//
//            }
//
//        }
//    }

    private class UnmarshallInlineTable implements Callable<InlineTable> {

        private final String xmlData;

        private final String inlineTableTag = "<InlineTable>";
        private final int inlineTableTagLength = inlineTableTag.length();

        private final String inlineTableEndTag = "</InlineTable>";
        private final int inlineTableEndTagLength = inlineTableEndTag.length();

        private final String rowTag = "<row>";
        private final int rowTagLength = rowTag.length();
        private final String rowEndTag = "</row>";
        private final int rowEndTagLength = rowEndTag.length();

        public UnmarshallInlineTable(String xmlData) {
            this.xmlData = xmlData;
        }

        @Override
        public InlineTable call() throws Exception {

            InlineTable table = unmarshalCustom();

            return table;
        }

        private InlineTable unmarshalCustom() {
            //ElementNSImpl - localName tag - namespaceUri "http://www.dmg.org/PMML-4_3" - firstChild TextImpl -> data - "A878"
            InlineTable table = new InlineTable();

            String rowsOnly = xmlData.substring(inlineTableTagLength, xmlData.length() - inlineTableEndTagLength);
            rowsOnly = rowsOnly.replaceAll("<content>", "").replaceAll("</content>", "");
            
            
            int offset = 0;

            while (offset < rowsOnly.length()) {

                int rowStartIndex = rowsOnly.indexOf(rowTag, offset);

                if (rowStartIndex == -1) {
                    break;
                }

                int rowEndIndex = rowsOnly.indexOf(rowEndTag, rowStartIndex + rowTagLength);

                int contentStartIndex = rowStartIndex + rowTagLength;
                int contentEndIndex = rowEndIndex;

                Row row = new Row();

                fillRow(row, rowsOnly.substring(contentStartIndex, contentEndIndex));

                table.addRows(row);

                offset = rowEndIndex + rowEndTagLength;
            }

            return table;

        }

        private void fillRow(Row row, String substring) {

            try {
                DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
                DocumentBuilder builder = dbf.newDocumentBuilder();
                Document parentDOM = builder.newDocument();

                int offset = 0;

                while (true) {

                    int startTagIndex = substring.indexOf("<", offset);
                    //end of string
                    if (startTagIndex == -1) {
                        break;
                    }
                    int endTagIndex = substring.indexOf(">", offset + 1);

                    String tagName;

                    int closeTagStart;
                    int closeTagEnd;
                    String tagContent;

                    //is this empty tag?
                    if (substring.charAt(endTagIndex - 1) == '/') {
                        tagName = substring.substring(startTagIndex + 1, endTagIndex - 1);

                        closeTagEnd = endTagIndex;
                        tagContent = "0";
                    } else {
                        tagName = substring.substring(startTagIndex + 1, endTagIndex);

                        closeTagStart = substring.indexOf("<", endTagIndex + 1);
                        closeTagEnd = substring.indexOf(">", endTagIndex + 1);
                        tagContent = substring.substring(endTagIndex + 1, closeTagStart);
                    }

                    offset = closeTagEnd;

                    Element parent = parentDOM.createElement(tagName);
                    parent.setTextContent(tagContent);
                    row.addContent(parent);
                }
            } catch (ParserConfigurationException ex) {
                Logger.getLogger(UnmarshallOptimized.class.getName()).log(Level.SEVERE, null, ex);
            }
        }
    }
}
